// Noise1234
// Author: Stefan Gustavson (stegu@itn.liu.se)
//
// This library is public domain software, released by the author
// into the public domain in February 2011. You may do anything
// you like with it. You may even remove all attributions,
// but of course I'd appreciate it if you kept my name somewhere.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.

/** \file
		\brief Implements the Noise1234 class for producing Perlin noise.
		\author Stefan Gustavson (stegu@itn.liu.se)
*/

/*
 * This implementation is "Improved Noise" as presented by
 * Ken Perlin at Siggraph 2002. The 3D function is a direct port
 * of his Java reference code available on www.noisemachine.com
 * (although I cleaned it up and made the code more readable),
 * but the 1D, 2D and 4D cases were implemented from scratch
 * by me.
 *
 * This is a highly reusable class. It has no dependencies
 * on any other file, apart from its own header file.
 */


//#include	"noise1234.h"

// This is the new and improved, C(2) continuous interpolant
#define FADE(t) ( t * t * t * ( t * ( t * 6 - 15 ) + 10 ) )

#define FASTFLOOR(x) ( ((x)>0) ? ((int)x) : ((int)x-1 ) )
#define LERP(t, a, b) ((a) + (t)*((b)-(a)))


//---------------------------------------------------------------------
// Static data

/*
 * Permutation table. This is just a random jumble of all numbers 0-255,
 * repeated twice to avoid wrapping the index at 255 for each lookup.
 * This needs to be exactly the same for all instances on all platforms,
 * so it's easiest to just keep it as static explicit data.
 * This also removes the need for any initialisation of this class.
 *
 * Note that making this an int[] instead of a char[] might make the
 * code run faster on platforms with a high penalty for unaligned single
 * byte addressing. Intel x86 is generally single-byte-friendly, but
 * some other CPUs are faster with 4-aligned reads.
 * However, a char[] is smaller, which avoids cache trashing, and that
 * is probably the most important aspect on most architectures.
 * This array is accessed a *lot* by the noise functions.
 * A vector-valued noise over 3D accesses it 96 times, and a
 * float-valued 4D noise 64 times. We want this to fit in the cache!
 */
unsigned char perm[512] = {9, 63, 240, 230, 95, 24, 244, 52, 149, 236, 32, 26,
175, 199, 4, 104, 125, 176, 160, 122, 184, 42, 103, 245, 43, 165, 164, 97,
172, 141, 190, 173, 153, 137, 131, 174, 30, 48, 86, 50, 217, 73, 61, 47, 68,
187, 70, 106, 55, 192, 179, 197, 229, 118, 226, 151, 114, 136, 154, 28, 7,
216, 33, 157, 200, 1, 213, 65, 40, 234, 66, 69, 2, 235, 11, 233, 3, 77, 225,
254, 146, 22, 150, 170, 18, 232, 121, 82, 111, 162, 60, 186, 19, 246, 78, 198,
147, 81, 130, 100, 201, 51, 53, 152, 238, 90, 20, 76, 166, 101, 221, 168, 110,
29, 218, 93, 171, 138, 252, 112, 202, 74, 220, 23, 209, 25, 204, 102, 247,
248, 193, 188, 143, 56, 214, 250, 107, 251, 181, 113, 206, 183, 167, 142, 94,
253, 16, 8, 64, 46, 211, 215, 44, 161, 249, 228, 117, 41, 92, 34, 99, 85, 156,
133, 31, 39, 6, 132, 126, 96, 14, 67, 180, 109, 189, 129, 224, 83, 140, 75,
178, 58, 134, 242, 177, 116, 27, 196, 195, 89, 79, 222, 227, 185, 241, 115, 49,
205, 108, 91, 35, 15, 210, 124, 72, 13, 243, 84, 208, 57, 155, 105, 158, 71,
10, 203, 239, 87, 148, 54, 139, 231, 169, 36, 219, 98, 38, 123, 45, 207, 144,
37, 212, 5, 163, 21, 194, 159, 88, 182, 17, 145, 62, 80, 119, 237, 223, 255,
128, 135, 127, 120, 0, 12, 59, 191,
9, 63, 240, 230, 95, 24, 244, 52, 149, 236, 32, 26,
175, 199, 4, 104, 125, 176, 160, 122, 184, 42, 103, 245, 43, 165, 164, 97,
172, 141, 190, 173, 153, 137, 131, 174, 30, 48, 86, 50, 217, 73, 61, 47, 68,
187, 70, 106, 55, 192, 179, 197, 229, 118, 226, 151, 114, 136, 154, 28, 7,
216, 33, 157, 200, 1, 213, 65, 40, 234, 66, 69, 2, 235, 11, 233, 3, 77, 225,
254, 146, 22, 150, 170, 18, 232, 121, 82, 111, 162, 60, 186, 19, 246, 78, 198,
147, 81, 130, 100, 201, 51, 53, 152, 238, 90, 20, 76, 166, 101, 221, 168, 110,
29, 218, 93, 171, 138, 252, 112, 202, 74, 220, 23, 209, 25, 204, 102, 247,
248, 193, 188, 143, 56, 214, 250, 107, 251, 181, 113, 206, 183, 167, 142, 94,
253, 16, 8, 64, 46, 211, 215, 44, 161, 249, 228, 117, 41, 92, 34, 99, 85, 156,
133, 31, 39, 6, 132, 126, 96, 14, 67, 180, 109, 189, 129, 224, 83, 140, 75,
178, 58, 134, 242, 177, 116, 27, 196, 195, 89, 79, 222, 227, 185, 241, 115, 49,
205, 108, 91, 35, 15, 210, 124, 72, 13, 243, 84, 208, 57, 155, 105, 158, 71,
10, 203, 239, 87, 148, 54, 139, 231, 169, 36, 219, 98, 38, 123, 45, 207, 144,
37, 212, 5, 163, 21, 194, 159, 88, 182, 17, 145, 62, 80, 119, 237, 223, 255,
128, 135, 127, 120, 0, 12, 59, 191};
/*
unsigned char perm2[] = {104, 180, 11, 23, 178, 173, 239, 202, 179, 101, 117,
122, 155, 33, 190, 222, 245, 208, 124, 135, 211, 160, 113, 22, 107, 188, 192,
243, 205, 90, 203, 47, 10, 48, 19, 231, 171, 186, 139, 141, 59, 43, 187, 54,
209, 99, 197, 100, 244, 87, 251, 219, 7, 156, 138, 53, 232, 238, 249, 50, 242,
68, 8, 39, 146, 162, 183, 1, 111, 74, 189, 94, 60, 148, 123, 5, 21, 213, 79,
254, 14, 35, 69, 91, 182, 37, 97, 159, 224, 112, 142, 165, 52, 3, 78, 227, 58,
109, 110, 168, 16, 46, 88, 126, 200, 184, 133, 44, 217, 61, 4, 12, 6, 115,
226, 204, 137, 29, 77, 214, 26, 175, 225, 45, 95, 147, 207, 105, 98, 82, 65,
151, 118, 81, 9, 240, 31, 185, 255, 199, 150, 212, 49, 194, 229, 103, 143, 89,
177, 161, 25, 210, 166, 252, 66, 0, 223, 86, 153, 246, 2, 106, 72, 247, 145,
253, 114, 116, 120, 83, 127, 233, 196, 57, 174, 198, 170, 27, 41, 70, 128, 20,
241, 71, 237, 215, 80, 236, 144, 92, 234, 169, 191, 181, 206, 167, 51, 221,
13, 40, 140, 56, 102, 73, 218, 62, 172, 216, 85, 30, 15, 136, 34, 176, 76, 93,
163, 164, 130, 36, 149, 157, 132, 248, 195, 108, 67, 119, 64, 154, 84, 220,
235, 129, 228, 96, 158, 230, 17, 152, 193, 32, 38, 125, 134, 63, 42, 24, 131,
28, 121, 201, 55, 250, 75, 18,
104, 180, 11, 23, 178, 173, 239, 202, 179, 101, 117,
122, 155, 33, 190, 222, 245, 208, 124, 135, 211, 160, 113, 22, 107, 188, 192,
243, 205, 90, 203, 47, 10, 48, 19, 231, 171, 186, 139, 141, 59, 43, 187, 54,
209, 99, 197, 100, 244, 87, 251, 219, 7, 156, 138, 53, 232, 238, 249, 50, 242,
68, 8, 39, 146, 162, 183, 1, 111, 74, 189, 94, 60, 148, 123, 5, 21, 213, 79,
254, 14, 35, 69, 91, 182, 37, 97, 159, 224, 112, 142, 165, 52, 3, 78, 227, 58,
109, 110, 168, 16, 46, 88, 126, 200, 184, 133, 44, 217, 61, 4, 12, 6, 115,
226, 204, 137, 29, 77, 214, 26, 175, 225, 45, 95, 147, 207, 105, 98, 82, 65,
151, 118, 81, 9, 240, 31, 185, 255, 199, 150, 212, 49, 194, 229, 103, 143, 89,
177, 161, 25, 210, 166, 252, 66, 0, 223, 86, 153, 246, 2, 106, 72, 247, 145,
253, 114, 116, 120, 83, 127, 233, 196, 57, 174, 198, 170, 27, 41, 70, 128, 20,
241, 71, 237, 215, 80, 236, 144, 92, 234, 169, 191, 181, 206, 167, 51, 221,
13, 40, 140, 56, 102, 73, 218, 62, 172, 216, 85, 30, 15, 136, 34, 176, 76, 93,
163, 164, 130, 36, 149, 157, 132, 248, 195, 108, 67, 119, 64, 154, 84, 220,
235, 129, 228, 96, 158, 230, 17, 152, 193, 32, 38, 125, 134, 63, 42, 24, 131,
28, 121, 201, 55, 250, 75, 18
};

unsigned char perm3[] = {213, 185, 110, 158, 86, 253, 68, 97, 87, 235, 31, 53,
90, 20, 114, 205, 248, 183, 54, 96, 172, 126, 26, 225, 108, 128, 33, 195, 212,
23, 153, 156, 50, 148, 247, 13, 150, 45, 67, 34, 1, 85, 136, 101, 120, 75,
177, 70, 119, 157, 100, 217, 124, 192, 3, 46, 164, 44, 92, 93, 43, 14, 228,
138, 58, 57, 19, 154, 47, 236, 107, 230, 220, 61, 203, 179, 21, 162, 166, 181,
161, 17, 4, 71, 255, 184, 171, 122, 73, 105, 0, 81, 232, 82, 223, 241, 38,
160, 62, 201, 64, 140, 133, 146, 238, 91, 249, 16, 254, 197, 174, 149, 72, 63,
155, 141, 102, 176, 118, 221, 244, 211, 237, 219, 116, 84, 80, 189, 252, 224,
39, 131, 125, 187, 29, 12, 48, 77, 175, 132, 245, 121, 231, 106, 42, 144, 51,
165, 137, 168, 9, 202, 109, 32, 127, 229, 216, 55, 69, 135, 60, 182, 24, 104,
147, 35, 88, 40, 233, 25, 52, 198, 83, 208, 7, 36, 65, 18, 170, 139, 145, 180,
27, 15, 207, 169, 123, 204, 2, 159, 167, 134, 103, 239, 8, 5, 111, 214, 186,
199, 56, 215, 227, 49, 94, 10, 76, 95, 89, 112, 113, 79, 41, 98, 240, 210,
190, 66, 196, 129, 130, 163, 11, 143, 59, 74, 218, 250, 226, 115, 193, 243,
246, 222, 6, 191, 28, 251, 152, 99, 30, 151, 173, 142, 234, 78, 117, 37, 188,
194, 209, 206, 22, 242, 200, 178,
213, 185, 110, 158, 86, 253, 68, 97, 87, 235, 31, 53,
90, 20, 114, 205, 248, 183, 54, 96, 172, 126, 26, 225, 108, 128, 33, 195, 212,
23, 153, 156, 50, 148, 247, 13, 150, 45, 67, 34, 1, 85, 136, 101, 120, 75,
177, 70, 119, 157, 100, 217, 124, 192, 3, 46, 164, 44, 92, 93, 43, 14, 228,
138, 58, 57, 19, 154, 47, 236, 107, 230, 220, 61, 203, 179, 21, 162, 166, 181,
161, 17, 4, 71, 255, 184, 171, 122, 73, 105, 0, 81, 232, 82, 223, 241, 38,
160, 62, 201, 64, 140, 133, 146, 238, 91, 249, 16, 254, 197, 174, 149, 72, 63,
155, 141, 102, 176, 118, 221, 244, 211, 237, 219, 116, 84, 80, 189, 252, 224,
39, 131, 125, 187, 29, 12, 48, 77, 175, 132, 245, 121, 231, 106, 42, 144, 51,
165, 137, 168, 9, 202, 109, 32, 127, 229, 216, 55, 69, 135, 60, 182, 24, 104,
147, 35, 88, 40, 233, 25, 52, 198, 83, 208, 7, 36, 65, 18, 170, 139, 145, 180,
27, 15, 207, 169, 123, 204, 2, 159, 167, 134, 103, 239, 8, 5, 111, 214, 186,
199, 56, 215, 227, 49, 94, 10, 76, 95, 89, 112, 113, 79, 41, 98, 240, 210,
190, 66, 196, 129, 130, 163, 11, 143, 59, 74, 218, 250, 226, 115, 193, 243,
246, 222, 6, 191, 28, 251, 152, 99, 30, 151, 173, 142, 234, 78, 117, 37, 188,
194, 209, 206, 22, 242, 200, 178
};

unsigned char perm4[] = {182, 94, 96, 219, 176, 38, 14, 42, 35, 218, 43, 163,
50, 151, 125, 74, 231, 78, 77, 221, 92, 233, 100, 21, 34, 251, 206, 210, 57,
255, 199, 153, 93, 83, 91, 165, 200, 234, 179, 250, 103, 82, 189, 205, 95, 81,
84, 236, 172, 195, 112, 30, 123, 8, 142, 152, 237, 230, 130, 169, 20, 215, 19,
198, 194, 188, 63, 224, 44, 145, 0, 192, 227, 131, 60, 160, 76, 157, 170, 243,
120, 45, 212, 107, 46, 147, 190, 178, 1, 25, 180, 67, 242, 59, 23, 138, 2,
126, 124, 228, 235, 102, 127, 101, 115, 87, 196, 240, 148, 238, 232, 187, 184,
197, 13, 113, 140, 26, 136, 71, 155, 203, 80, 29, 167, 252, 88, 55, 18, 114,
168, 119, 11, 54, 211, 181, 146, 106, 149, 31, 217, 39, 10, 144, 66, 110, 186,
86, 156, 36, 185, 3, 249, 27, 216, 33, 154, 61, 53, 166, 32, 116, 245, 99,
128, 90, 213, 193, 137, 133, 207, 204, 52, 175, 132, 79, 248, 51, 118, 5, 150,
158, 161, 159, 209, 15, 173, 62, 68, 28, 108, 64, 98, 183, 164, 56, 134, 202,
214, 48, 229, 12, 65, 223, 72, 239, 69, 9, 208, 220, 41, 47, 241, 141, 117,
85, 24, 201, 222, 171, 246, 253, 37, 97, 58, 135, 247, 49, 73, 6, 40, 129,
191, 17, 122, 244, 226, 111, 16, 177, 162, 254, 104, 174, 139, 143, 121, 70,
22, 75, 4, 89, 105, 225, 109, 7,
182, 94, 96, 219, 176, 38, 14, 42, 35, 218, 43, 163,
50, 151, 125, 74, 231, 78, 77, 221, 92, 233, 100, 21, 34, 251, 206, 210, 57,
255, 199, 153, 93, 83, 91, 165, 200, 234, 179, 250, 103, 82, 189, 205, 95, 81,
84, 236, 172, 195, 112, 30, 123, 8, 142, 152, 237, 230, 130, 169, 20, 215, 19,
198, 194, 188, 63, 224, 44, 145, 0, 192, 227, 131, 60, 160, 76, 157, 170, 243,
120, 45, 212, 107, 46, 147, 190, 178, 1, 25, 180, 67, 242, 59, 23, 138, 2,
126, 124, 228, 235, 102, 127, 101, 115, 87, 196, 240, 148, 238, 232, 187, 184,
197, 13, 113, 140, 26, 136, 71, 155, 203, 80, 29, 167, 252, 88, 55, 18, 114,
168, 119, 11, 54, 211, 181, 146, 106, 149, 31, 217, 39, 10, 144, 66, 110, 186,
86, 156, 36, 185, 3, 249, 27, 216, 33, 154, 61, 53, 166, 32, 116, 245, 99,
128, 90, 213, 193, 137, 133, 207, 204, 52, 175, 132, 79, 248, 51, 118, 5, 150,
158, 161, 159, 209, 15, 173, 62, 68, 28, 108, 64, 98, 183, 164, 56, 134, 202,
214, 48, 229, 12, 65, 223, 72, 239, 69, 9, 208, 220, 41, 47, 241, 141, 117,
85, 24, 201, 222, 171, 246, 253, 37, 97, 58, 135, 247, 49, 73, 6, 40, 129,
191, 17, 122, 244, 226, 111, 16, 177, 162, 254, 104, 174, 139, 143, 121, 70,
22, 75, 4, 89, 105, 225, 109, 7
};
*/

//---------------------------------------------------------------------

/*
 * Helper functions to compute gradients-dot-residualvectors (1D to 4D)
 * Note that these generate gradients of more than unit length. To make
 * a close match with the value range of classic Perlin noise, the final
 * noise values need to be rescaled. To match the RenderMan noise in a
 * statistical sense, the approximate scaling values (empirically
 * determined from test renderings) are:
 * 1D noise needs rescaling with 0.188
 * 2D noise needs rescaling with 0.507
 * 3D noise needs rescaling with 0.936
 * 4D noise needs rescaling with 0.87
 * Note that these noise functions are the most practical and useful
 * signed version of Perlin noise. To return values according to the
 * RenderMan specification from the SL noise() and pnoise() functions,
 * the noise values need to be scaled and offset to [0,1], like this:
 * float SLnoise = (Noise1234::noise(x,y,z) + 1.0) * 0.5;
 */

float  grad1d( int hash, float x ) {
    int h = hash & 15;
    float grad = 1.0 + (h & 7);  // Gradient value 1.0, 2.0, ..., 8.0
    if (h&8) grad = -grad;         // and a random sign for the gradient
    return ( grad * x );           // Multiply the gradient with the distance
}

float  grad2d( int hash, float x, float y ) {
    int h = hash & 7;      // Convert low 3 bits of hash code
    float u = h<4 ? x : y;  // into 8 simple gradient directions,
    float v = h<4 ? y : x;  // and compute the dot product with (x,y).
    return ((h&1)? -u : u) + ((h&2)? -2.0*v : 2.0*v);
}

float  grad3d( int hash, float x, float y , float z ) {
    int h = hash & 15;     // Convert low 4 bits of hash code into 12 simple
    float u = h<8 ? x : y; // gradient directions, and compute dot product.
    float v = h<4 ? y : h==12||h==14 ? x : z; // Fix repeats at h = 12 to 15
    return ((h&1)? -u : u) + ((h&2)? -v : v);
}

float  grad4d( int hash, float x, float y, float z, float t ) {
    int h = hash & 31;      // Convert low 5 bits of hash code into 32 simple
    float u = h<24 ? x : y; // gradient directions, and compute dot product.
    float v = h<16 ? y : z;
    float w = h<8 ? z : t;
    return ((h&1)? -u : u) + ((h&2)? -v : v) + ((h&4)? -w : w);
}

int  fastfloorf ( double x) {
    return ( ((x)>0) ? ((int)x) : ((int)x-1 ) );
}
long int  fastmod (int x, int y) {
    return ( x>0 ? x%y : y-(-x%y) );
}
//---------------------------------------------------------------------
/** 1D float Perlin noise, SL "noise()"
 */
float noise1d( float x, int seed )
{
    int ix0, ix1;
    float fx0, fx1;
    float s, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    fx0 = x - ix0;       // Fractional part of x
    fx1 = fx0 - 1.0f;
    ix1 = ( ix0+1 ) & 0xff;
    ix0 = ix0 & 0xff;    // Wrap to 0..255

    s = FADE( fx0 );

    n0 = grad1d( perm[ ix0 + perm[seed] ], fx0 );
    n1 = grad1d( perm[ ix1 + perm[seed] ], fx1 );
    return 0.188f * ( LERP( s, n0, n1 ) );
}

//---------------------------------------------------------------------
/** 1D float Perlin periodic noise, SL "pnoise()"
 */
float pnoise1d( float x, int px, int seed )
{
    int ix0, ix1;
    float fx0, fx1;
    float s, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    fx0 = x - ix0;       // Fractional part of x
    fx1 = fx0 - 1.0f;
    ix1 = (( ix0 + 1 ) % px) & 0xff; // Wrap to 0..px-1 *and* wrap to 0..255
    ix0 = ( ix0 % px ) & 0xff;      // (because px might be greater than 256)

    s = FADE( fx0 );

    n0 = grad1d( perm[ ix0 + perm[seed] ], fx0 );
    n1 = grad1d( perm[ ix1 + perm[seed] ], fx1 );
    return 0.188f * ( LERP( s, n0, n1 ) );
}


//---------------------------------------------------------------------
/** 2D float Perlin noise.
 */
float noise2d( float x, float y, int seed )
{
    int ix0, iy0, ix1, iy1;
    long int ixq0, iyq0, ixq1, iyq1;
    float fx0, fy0, fx1, fy1;
    float s, t, nx0, nx1, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    iy0 = FASTFLOOR( y ); // Integer part of y
    ixq0 = (fastmod( ix0 , 65536 ) >> 8 ) & 0xff;
    iyq0 = (fastmod( iy0 , 65536 ) >> 8 ) & 0xff;
    ixq1 = (fastmod( ix0 + 1 , 65536 ) >> 8) & 0xff;
    iyq1 = (fastmod( iy0 + 1 , 65536 ) >> 8) & 0xff;
    fx0 = x - ix0;        // Fractional part of x
    fy0 = y - iy0;        // Fractional part of y
    fx1 = fx0 - 1.0f;
    fy1 = fy0 - 1.0f;
    ix1 = fastmod( ix0 + 1 , 256);  // Wrap to 0..255
    iy1 = fastmod( iy0 + 1 , 256);  // Wrap to 0..255
    ix0 = fastmod( ix0 , 256);
    iy0 = fastmod( iy0 , 256);

    t = FADE( fy0 );
    s = FADE( fx0 );

    nx0 = grad2d(perm[ix0 + perm[ iy0 + perm[ ixq0 + perm[ iyq0 + perm[seed] ] ] ] ], fx0, fy0);
    nx1 = grad2d(perm[ix0 + perm[ iy1 + perm[ ixq0 + perm[ iyq1 + perm[seed] ] ] ] ], fx0, fy1);
//    nx0 = grad2d(perm[ix0 + perm[ iy0 + perm[ 0 + perm[ 0 + perm[seed] ] ] ] ], fx0, fy0);
//    nx1 = grad2d(perm[ix0 + perm[ iy1 + perm[ 0 + perm[ 0 + perm[seed] ] ] ] ], fx0, fy1);
    n0 = LERP( t, nx0, nx1 );

    nx0 = grad2d(perm[ix1 + perm[ iy0 + perm[ ixq1 + perm[ iyq0 + perm[seed] ] ] ] ], fx1, fy0);
    nx1 = grad2d(perm[ix1 + perm[ iy1 + perm[ ixq1 + perm[ iyq1 + perm[seed] ] ] ] ], fx1, fy1);
//    nx0 = grad2d(perm[ix1 + perm[ iy0 + perm[ 0 + perm[ 0 + perm[seed] ] ] ] ], fx1, fy0);
//    nx1 = grad2d(perm[ix1 + perm[ iy1 + perm[ 0 + perm[ 0 + perm[seed] ] ] ] ], fx1, fy1);
    n1 = LERP(t, nx0, nx1);

    return 0.507f * ( LERP( s, n0, n1 ) );
}

//---------------------------------------------------------------------
/** 2D float Perlin periodic noise.
 */
float pnoise2d( float x, float y, int px, int py, int seed )
{
    int ix0, iy0, ix1, iy1;
    float fx0, fy0, fx1, fy1;
    float s, t, nx0, nx1, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    iy0 = FASTFLOOR( y ); // Integer part of y
    fx0 = x - ix0;        // Fractional part of x
    fy0 = y - iy0;        // Fractional part of y
    fx1 = fx0 - 1.0f;
    fy1 = fy0 - 1.0f;
    ix1 = (( ix0 + 1 ) % px) & 0xff;  // Wrap to 0..px-1 and wrap to 0..255
    iy1 = (( iy0 + 1 ) % py) & 0xff;  // Wrap to 0..py-1 and wrap to 0..255
    ix0 = ( ix0 % px ) & 0xff;
    iy0 = ( iy0 % py ) & 0xff;

    t = FADE( fy0 );
    s = FADE( fx0 );

    nx0 = grad2d(perm[ix0 + perm[iy0 + perm[seed]]], fx0, fy0);
    nx1 = grad2d(perm[ix0 + perm[iy1 + perm[seed]]], fx0, fy1);
    n0 = LERP( t, nx0, nx1 );

    nx0 = grad2d(perm[ix1 + perm[iy0 + perm[seed]]], fx1, fy0);
    nx1 = grad2d(perm[ix1 + perm[iy1 + perm[seed]]], fx1, fy1);
    n1 = LERP(t, nx0, nx1);

    return 0.507f * ( LERP( s, n0, n1 ) );
}


//---------------------------------------------------------------------
/** 3D float Perlin noise.
 */
float noise3d( float x, float y, float z, int seed )
{
    int ix0, iy0, ix1, iy1, iz0, iz1;
    float fx0, fy0, fz0, fx1, fy1, fz1;
    float s, t, r;
    float nxy0, nxy1, nx0, nx1, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    iy0 = FASTFLOOR( y ); // Integer part of y
    iz0 = FASTFLOOR( z ); // Integer part of z
    fx0 = x - ix0;        // Fractional part of x
    fy0 = y - iy0;        // Fractional part of y
    fz0 = z - iz0;        // Fractional part of z
    fx1 = fx0 - 1.0f;
    fy1 = fy0 - 1.0f;
    fz1 = fz0 - 1.0f;
    ix1 = ( ix0 + 1 ) & 0xff; // Wrap to 0..255
    iy1 = ( iy0 + 1 ) & 0xff;
    iz1 = ( iz0 + 1 ) & 0xff;
    ix0 = ix0 & 0xff;
    iy0 = iy0 & 0xff;
    iz0 = iz0 & 0xff;

    r = FADE( fz0 );
    t = FADE( fy0 );
    s = FADE( fx0 );

    nxy0 = grad3d(perm[ix0 + perm[iy0 + perm[iz0 + perm[seed]]]], fx0, fy0, fz0);
    nxy1 = grad3d(perm[ix0 + perm[iy0 + perm[iz1 + perm[seed]]]], fx0, fy0, fz1);
    nx0 = LERP( r, nxy0, nxy1 );

    nxy0 = grad3d(perm[ix0 + perm[iy1 + perm[iz0 + perm[seed]]]], fx0, fy1, fz0);
    nxy1 = grad3d(perm[ix0 + perm[iy1 + perm[iz1 + perm[seed]]]], fx0, fy1, fz1);
    nx1 = LERP( r, nxy0, nxy1 );

    n0 = LERP( t, nx0, nx1 );

    nxy0 = grad3d(perm[ix1 + perm[iy0 + perm[iz0 + perm[seed]]]], fx1, fy0, fz0);
    nxy1 = grad3d(perm[ix1 + perm[iy0 + perm[iz1 + perm[seed]]]], fx1, fy0, fz1);
    nx0 = LERP( r, nxy0, nxy1 );

    nxy0 = grad3d(perm[ix1 + perm[iy1 + perm[iz0 + perm[seed]]]], fx1, fy1, fz0);
    nxy1 = grad3d(perm[ix1 + perm[iy1 + perm[iz1 + perm[seed]]]], fx1, fy1, fz1);
    nx1 = LERP( r, nxy0, nxy1 );

    n1 = LERP( t, nx0, nx1 );

    return 0.936f * ( LERP( s, n0, n1 ) );
}

//---------------------------------------------------------------------
/** 3D float Perlin periodic noise.
 */
float pnoise3d( float x, float y, float z, int px, int py, int pz, int seed )
{
    int ix0, iy0, ix1, iy1, iz0, iz1;
    float fx0, fy0, fz0, fx1, fy1, fz1;
    float s, t, r;
    float nxy0, nxy1, nx0, nx1, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    iy0 = FASTFLOOR( y ); // Integer part of y
    iz0 = FASTFLOOR( z ); // Integer part of z
    fx0 = x - ix0;        // Fractional part of x
    fy0 = y - iy0;        // Fractional part of y
    fz0 = z - iz0;        // Fractional part of z
    fx1 = fx0 - 1.0f;
    fy1 = fy0 - 1.0f;
    fz1 = fz0 - 1.0f;
    ix1 = (( ix0 + 1 ) % px ) & 0xff; // Wrap to 0..px-1 and wrap to 0..255
    iy1 = (( iy0 + 1 ) % py ) & 0xff; // Wrap to 0..py-1 and wrap to 0..255
    iz1 = (( iz0 + 1 ) % pz ) & 0xff; // Wrap to 0..pz-1 and wrap to 0..255
    ix0 = ( ix0 % px ) & 0xff;
    iy0 = ( iy0 % py ) & 0xff;
    iz0 = ( iz0 % pz ) & 0xff;

    r = FADE( fz0 );
    t = FADE( fy0 );
    s = FADE( fx0 );

    nxy0 = grad3d(perm[ix0 + perm[iy0 + perm[iz0 + perm[seed]]]], fx0, fy0, fz0);
    nxy1 = grad3d(perm[ix0 + perm[iy0 + perm[iz1 + perm[seed]]]], fx0, fy0, fz1);
    nx0 = LERP( r, nxy0, nxy1 );

    nxy0 = grad3d(perm[ix0 + perm[iy1 + perm[iz0 + perm[seed]]]], fx0, fy1, fz0);
    nxy1 = grad3d(perm[ix0 + perm[iy1 + perm[iz1 + perm[seed]]]], fx0, fy1, fz1);
    nx1 = LERP( r, nxy0, nxy1 );

    n0 = LERP( t, nx0, nx1 );

    nxy0 = grad3d(perm[ix1 + perm[iy0 + perm[iz0 + perm[seed]]]], fx1, fy0, fz0);
    nxy1 = grad3d(perm[ix1 + perm[iy0 + perm[iz1 + perm[seed]]]], fx1, fy0, fz1);
    nx0 = LERP( r, nxy0, nxy1 );

    nxy0 = grad3d(perm[ix1 + perm[iy1 + perm[iz0 + perm[seed]]]], fx1, fy1, fz0);
    nxy1 = grad3d(perm[ix1 + perm[iy1 + perm[iz1 + perm[seed]]]], fx1, fy1, fz1);
    nx1 = LERP( r, nxy0, nxy1 );

    n1 = LERP( t, nx0, nx1 );

    return 0.936f * ( LERP( s, n0, n1 ) );
}


//---------------------------------------------------------------------
/** 4D float Perlin noise.
 */

float noise4d( float x, float y, float z, float w, int seed )
{
    int ix0, iy0, iz0, iw0, ix1, iy1, iz1, iw1;
    float fx0, fy0, fz0, fw0, fx1, fy1, fz1, fw1;
    float s, t, r, q;
    float nxyz0, nxyz1, nxy0, nxy1, nx0, nx1, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    iy0 = FASTFLOOR( y ); // Integer part of y
    iz0 = FASTFLOOR( z ); // Integer part of y
    iw0 = FASTFLOOR( w ); // Integer part of w
    fx0 = x - ix0;        // Fractional part of x
    fy0 = y - iy0;        // Fractional part of y
    fz0 = z - iz0;        // Fractional part of z
    fw0 = w - iw0;        // Fractional part of w
    fx1 = fx0 - 1.0f;
    fy1 = fy0 - 1.0f;
    fz1 = fz0 - 1.0f;
    fw1 = fw0 - 1.0f;
    ix1 = ( ix0 + 1 ) & 0xff;  // Wrap to 0..255
    iy1 = ( iy0 + 1 ) & 0xff;
    iz1 = ( iz0 + 1 ) & 0xff;
    iw1 = ( iw0 + 1 ) & 0xff;
    ix0 = ix0 & 0xff;
    iy0 = iy0 & 0xff;
    iz0 = iz0 & 0xff;
    iw0 = iw0 & 0xff;

    q = FADE( fw0 );
    r = FADE( fz0 );
    t = FADE( fy0 );
    s = FADE( fx0 );

    nxyz0 = grad4d(perm[ix0 + perm[iy0 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx0, fy0, fz0, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy0 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx0, fy0, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix0 + perm[iy0 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx0, fy0, fz1, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy0 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx0, fy0, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx0 = LERP ( r, nxy0, nxy1 );

    nxyz0 = grad4d(perm[ix0 + perm[iy1 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx0, fy1, fz0, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy1 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx0, fy1, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix0 + perm[iy1 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx0, fy1, fz1, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy1 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx0, fy1, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx1 = LERP ( r, nxy0, nxy1 );

    n0 = LERP( t, nx0, nx1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy0 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx1, fy0, fz0, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy0 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx1, fy0, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy0 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx1, fy0, fz1, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy0 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx1, fy0, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx0 = LERP ( r, nxy0, nxy1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy1 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx1, fy1, fz0, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy1 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx1, fy1, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy1 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx1, fy1, fz1, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy1 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx1, fy1, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx1 = LERP ( r, nxy0, nxy1 );

    n1 = LERP( t, nx0, nx1 );

    return 0.87f * ( LERP( s, n0, n1 ) );
}

//---------------------------------------------------------------------
/** 4D float Perlin periodic noise.
 */

float pnoise4d( float x, float y, float z, float w,
                            int px, int py, int pz, int pw, int seed )
{
    int ix0, iy0, iz0, iw0, ix1, iy1, iz1, iw1;
    float fx0, fy0, fz0, fw0, fx1, fy1, fz1, fw1;
    float s, t, r, q;
    float nxyz0, nxyz1, nxy0, nxy1, nx0, nx1, n0, n1;

    ix0 = FASTFLOOR( x ); // Integer part of x
    iy0 = FASTFLOOR( y ); // Integer part of y
    iz0 = FASTFLOOR( z ); // Integer part of y
    iw0 = FASTFLOOR( w ); // Integer part of w
    fx0 = x - ix0;        // Fractional part of x
    fy0 = y - iy0;        // Fractional part of y
    fz0 = z - iz0;        // Fractional part of z
    fw0 = w - iw0;        // Fractional part of w
    fx1 = fx0 - 1.0f;
    fy1 = fy0 - 1.0f;
    fz1 = fz0 - 1.0f;
    fw1 = fw0 - 1.0f;
    ix1 = (( ix0 + 1 ) % px ) & 0xff;  // Wrap to 0..px-1 and wrap to 0..255
    iy1 = (( iy0 + 1 ) % py ) & 0xff;  // Wrap to 0..py-1 and wrap to 0..255
    iz1 = (( iz0 + 1 ) % pz ) & 0xff;  // Wrap to 0..pz-1 and wrap to 0..255
    iw1 = (( iw0 + 1 ) % pw ) & 0xff;  // Wrap to 0..pw-1 and wrap to 0..255
    ix0 = ( ix0 % px ) & 0xff;
    iy0 = ( iy0 % py ) & 0xff;
    iz0 = ( iz0 % pz ) & 0xff;
    iw0 = ( iw0 % pw ) & 0xff;

    q = FADE( fw0 );
    r = FADE( fz0 );
    t = FADE( fy0 );
    s = FADE( fx0 );

    nxyz0 = grad4d(perm[ix0 + perm[iy0 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx0, fy0, fz0, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy0 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx0, fy0, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix0 + perm[iy0 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx0, fy0, fz1, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy0 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx0, fy0, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx0 = LERP ( r, nxy0, nxy1 );

    nxyz0 = grad4d(perm[ix0 + perm[iy1 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx0, fy1, fz0, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy1 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx0, fy1, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix0 + perm[iy1 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx0, fy1, fz1, fw0);
    nxyz1 = grad4d(perm[ix0 + perm[iy1 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx0, fy1, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx1 = LERP ( r, nxy0, nxy1 );

    n0 = LERP( t, nx0, nx1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy0 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx1, fy0, fz0, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy0 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx1, fy0, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy0 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx1, fy0, fz1, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy0 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx1, fy0, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx0 = LERP ( r, nxy0, nxy1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy1 + perm[iz0 + perm[iw0 + perm[seed]]]]], fx1, fy1, fz0, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy1 + perm[iz0 + perm[iw1 + perm[seed]]]]], fx1, fy1, fz0, fw1);
    nxy0 = LERP( q, nxyz0, nxyz1 );

    nxyz0 = grad4d(perm[ix1 + perm[iy1 + perm[iz1 + perm[iw0 + perm[seed]]]]], fx1, fy1, fz1, fw0);
    nxyz1 = grad4d(perm[ix1 + perm[iy1 + perm[iz1 + perm[iw1 + perm[seed]]]]], fx1, fy1, fz1, fw1);
    nxy1 = LERP( q, nxyz0, nxyz1 );

    nx1 = LERP ( r, nxy0, nxy1 );

    n1 = LERP( t, nx0, nx1 );

    return 0.87f * ( LERP( s, n0, n1 ) );
}

//---------------------------------------------------------------------
